using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.AccessControl;
using System.Security.Principal;
using System.Text;

namespace DuplicateFinder
{
    class Program
    {
        static void Main(string[] args)
        {
            bool recurseIntoSubdirectories = false;

            if (args.Length < 1)
            {
                ShowUsage();
                return;
            }

            int firstDirectoryIndex = 0;
            IEnumerable<string> directoriesToSearch = null;
            bool testDirectoriesMade = false;

            try
            {
                // Sprawdzamy, czy program działa w trybie testowym.
                if (args.Length == 1 && args[0] == "/test")
                {
                    directoriesToSearch = MakeTestDirectories();
                    testDirectoriesMade = true;
                    recurseIntoSubdirectories = true;
                }
                else
                {
                    if (args.Length > 1)
                    {
                        // Sprawdzamy, czy katalogi mają być sprawdzane rekurencyjnie.
                        if (args[0] == "/sub")
                        {
                            if (args.Length < 2)
                            {
                                ShowUsage();
                                return;
                            }
                            recurseIntoSubdirectories = true;
                            firstDirectoryIndex = 1;
                        }
                    }

                    // Pobranie listy katalogów z wiersza poleceń.
                    directoriesToSearch = args.Skip(firstDirectoryIndex);
                }
                List<FileNameGroup> filesGroupedByName =
                    InspectDirectories(recurseIntoSubdirectories, directoriesToSearch);

                DisplayMatches(filesGroupedByName);
                Console.ReadKey();
            }
            finally
            {
                if (testDirectoriesMade)
                {
                    CleanupTestDirectories(directoriesToSearch);
                }
            }
        }

        private static void ShowUsage()
        {
            Console.WriteLine("Odnajdywanie duplikatów plików.");
            Console.WriteLine("====================");
            Console.WriteLine(
                "Program poszukuje powtarzających się plików w jednym lub kilku katalogach.");
            Console.WriteLine();
            Console.WriteLine(
                "Sposób korzystania: findduplicatefiles [/sub] NazwaKatalogu [NazwaKatalogu] ...");
            Console.WriteLine("/sub - rekurencyjne przeszukiwanie podkatalogów.");
            Console.ReadKey();
        }

        private static List<FileNameGroup> InspectDirectories(
            bool recurseIntoSubdirectories,
            IEnumerable<string> directoriesToSearch)
        {
            var searchOption = recurseIntoSubdirectories ?
                SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;

            // Pobranie ścieżki dostępu do każdego pliku w każdym 
            // z przeszukiwanych katalogów
            var allFilePaths = from directory in directoriesToSearch
                               from file in Directory.GetFiles(directory, "*.*",
                                                                searchOption)
                               select file;

            // Pogrupowanie plików na podstawie lokalnej nazwy (czyli nazwy bez
            // ścieżki dostępu do katalogu, w jakim plik się znajduje) oraz utworzenie 
            // dla każdej nazwy pliku listy zawierającej szczegółowe informacje 
            // dotyczące każdego pliku o tej nazwie  
            var fileNameGroups = from filePath in allFilePaths
                                 let fileNameWithoutPath = Path.GetFileName(filePath)
                                 group filePath by fileNameWithoutPath into nameGroup
                                 select new FileNameGroup
                                 {
                                     FileNameWithoutPath = nameGroup.Key,
                                     FilesWithThisName =
                                      (from filePath in nameGroup
                                       let info = new FileInfo(filePath)
                                       select new FileDetails
                                       {
                                           FilePath = filePath,
                                           FileSize = info.Length
                                       }).ToList()
                                 };

            return fileNameGroups.ToList();
        }

        private static void DisplayMatches(
            IEnumerable<FileNameGroup> filesGroupedByName)
        {
            var groupsWithMoreThanOneFile = from nameGroup in filesGroupedByName
                                            where nameGroup.FilesWithThisName.Count > 1
                                            select nameGroup;

            foreach (var fileNameGroup in groupsWithMoreThanOneFile)
            {
                // Pogrupowanie odpowiadających sobie plików na podstawie ich wielkości
                // i wybranie tych, których dla danej wielkości jest więcej niż jeden 
                var matchesBySize = from file in fileNameGroup.FilesWithThisName
                                    group file by file.FileSize into sizeGroup
                                    where sizeGroup.Count() > 1
                                    select sizeGroup;

                foreach (var matchedBySize in matchesBySize)
                {
                    string fileNameAndSize = string.Format("{0} ({1} bytes)",
                    fileNameGroup.FileNameWithoutPath, matchedBySize.Key);
                    WriteWithUnderlines(fileNameAndSize);
                    // Wyświetlenie każdego katalogu zawierającego dany plik
                    foreach (var file in matchedBySize)
                    {
                        Console.WriteLine(Path.GetDirectoryName(file.FilePath));
                    }
                    Console.WriteLine();
                }
            }
        }

        private static void WriteWithUnderlines(string text)
        {
            Console.WriteLine(text);
            Console.WriteLine(new string('-', text.Length));
        }

        // Listing 11-15. Konfiguracja kontroli dostępu do nowych katalogów.
        private static string[] MakeTestDirectories()
        {
            string localApplicationData = Path.Combine(
                Environment.GetFolderPath(
                    Environment.SpecialFolder.LocalApplicationData),
                @"Programming CSharp\FindDuplicates");

            // Pobranie nazwy bieżącego użytkownika
            string userName = WindowsIdentity.GetCurrent().Name;
            // Określenie reguły kontroli dostępu
            FileSystemAccessRule fsarAllow =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.FullControl,
                    AccessControlType.Allow);
            DirectorySecurity ds = new DirectorySecurity();
            ds.AddAccessRule(fsarAllow);

            // Listing 11-16. Odbieranie uprawnień.
            FileSystemAccessRule fsarDeny =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.WriteExtendedAttributes,
                    AccessControlType.Deny);
            ds.AddAccessRule(fsarDeny);

            // Tworzymy trzy katalogi testowe.
            var directories = new string[3];
            for (int i = 0; i < directories.Length; ++i)
            {
                string directory = Path.GetRandomFileName();
                // Łączymy dane lokalnej aplikacji z losowymi 
                // nazwami plików i katalogów...
                string fullPath = Path.Combine(localApplicationData, directory);
                // I tworzymy katalog.
                Directory.CreateDirectory(fullPath, ds);
                directories[i] = fullPath;
                Console.WriteLine(fullPath);
            }
            // Listing 11-18. Tworzenie plików w katalogach testowych.
            CreateTestFiles(directories);
            return directories;
        }

        // Listing 11-17. Usuwanie katalogu.
        private static void CleanupTestDirectories(IEnumerable<string> directories)
        {
            foreach (var directory in directories)
            {
                Directory.Delete(directory);
            }
        }

        // Listing 11-19. Metoda CreateTestFiles.
        private static void CreateTestFiles(IEnumerable<string> directories)
        {
            string fileForAllDirectories = "SameNameAndContent.txt";
            string fileSameInAllButDifferentSizes = "SameNameDifferentSize.txt";
            int directoryIndex = 0;
            // Tworzymy plik, który zostanie umieszczony w każdym katalogu.
            foreach (string directory in directories)
            {
                directoryIndex++;

                // Tworzymy unikalny plik dla danego katalogu.
                string filename = Path.GetRandomFileName();
                string fullPath = Path.Combine(directory, filename);
                CreateFile(fullPath, "Przykładowa zawartość nr 1");

                // A teraz plik, który będzie się pojawiał we wszystkich katalogach
                // i będzie miał tę samą zawartość.
                fullPath = Path.Combine(directory, fileForAllDirectories);
                CreateFile(fullPath, "Jestem w każdym katalogu");

                // I kolejny plik, który w każdym katalogu będzie miał taką samą
                // nazwę, lecz inną zawartość.
                fullPath = Path.Combine(directory, fileSameInAllButDifferentSizes);
                StringBuilder builder = new StringBuilder();
                builder.AppendLine("Nowa zawartość: ");
                builder.AppendLine(new string('x', directoryIndex));
                CreateFile(fullPath, builder.ToString());
            }
        }

        // Listing 11-20. Zapis łańcucha znaków do nowego pliku.
        private static void CreateFile(string fullPath, string contents)
        {
            File.WriteAllText(fullPath, contents);
        }
    }
}
